<?php
// page2.php — Kardeşler Terzi (Excel benzeri: Gönderiler + Tahsilatlar + Bakiye)
declare(strict_types=1);
session_start();
require_once __DIR__ . '/db.php';

if (empty($_SESSION['user'])) { header('Location: login.php'); exit; }

// CSRF
if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); }
$csrf = $_SESSION['csrf_token'];
$user = $_SESSION['user'];

// -------------------- SAVE endpoint (JSON) --------------------
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_GET['action'] ?? '')==='save') {
  if (!isset($_SERVER['HTTP_X_CSRF_TOKEN']) || !hash_equals($csrf, (string)$_SERVER['HTTP_X_CSRF_TOKEN'])) {
    http_response_code(403); echo json_encode(['ok'=>false,'msg'=>'CSRF']); exit;
  }
  $raw = file_get_contents('php://input');
  $data = json_decode($raw, true);
  if (!is_array($data)) { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Bad JSON']); exit; }

  $entries  = $data['entries']  ?? [];  // üst tablo
  $receipts = $data['receipts'] ?? [];  // alt tablo

  // Basit validasyon + normalize
  $safeEntries = [];
  foreach ($entries as $r) {
    $d = $r['date'] ?? '';
    $sent = trim((string)($r['sent'] ?? ''));
    $qty = (float)($r['qty'] ?? 0);
    $price = (float)($r['price'] ?? 0);
    $amount = round($qty * $price, 2);
    if ($d && $sent !== '' && $qty > 0) {
      $safeEntries[] = [$d, $sent, $qty, $price, $amount];
    }
  }
  $safeReceipts = [];
  foreach ($receipts as $r) {
    $d = $r['date'] ?? '';
    $desc = trim((string)($r['desc'] ?? ''));
    $amt = (float)($r['amount'] ?? 0);
    if ($d && $desc !== '' && $amt > 0) {
      $safeReceipts[] = [$d, $desc, $amt];
    }
  }

  // Transaction: ikisini de sil → yeniden ekle
  $pdo->beginTransaction();
  try {
    $pdo->exec("DELETE FROM tailor_entries");
    $pdo->exec("DELETE FROM tailor_receipts");

    if ($safeEntries) {
      $st = $pdo->prepare("INSERT INTO tailor_entries (t_date, sent_item, qty, price, amount) VALUES (?,?,?,?,?)");
      foreach ($safeEntries as $row) { $st->execute($row); }
    }
    if ($safeReceipts) {
      $st = $pdo->prepare("INSERT INTO tailor_receipts (r_date, description, amount) VALUES (?,?,?)");
      foreach ($safeReceipts as $row) { $st->execute($row); }
    }

    $pdo->commit();
    echo json_encode(['ok'=>true]);
  } catch (Throwable $e) {
    $pdo->rollBack();
    http_response_code(500);
    echo json_encode(['ok'=>false,'msg'=>'DB error']);
  }
  exit;
}

// -------------------- Sayfa verileri --------------------
$entries = $pdo->query("SELECT t_date AS date, sent_item AS sent, qty, price, amount FROM tailor_entries ORDER BY t_date ASC, id ASC")->fetchAll();
$receipts = $pdo->query("SELECT r_date AS date, description AS `desc`, amount FROM tailor_receipts ORDER BY r_date ASC, id ASC")->fetchAll();
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Kardeşler Terzi</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
<style>
  body{background:#f7fafc;}
  .sheet-card{border-radius:16px;border:1px solid rgba(0,0,0,.06);box-shadow:0 10px 30px rgba(2,8,23,.06);}
  .zone-title{font-weight:700;letter-spacing:.3px}
  .grid th, .grid td{vertical-align:middle}
  .grid thead th{background:#e5e7eb; position:sticky; top:0; z-index:2}
  .grid input{border:0;background:transparent; width:100%; padding:.35rem .5rem; outline:none}
  .grid input:focus{background:#fffbe6; box-shadow: inset 0 0 0 1px #f59e0b; border-radius:8px}
  .section-head{background:#e5e7eb;border-radius:10px;padding:.3rem .6rem;display:inline-block}
  .totals{font-weight:700}
</style>
</head>
<body>
<nav class="navbar navbar-light bg-white border-bottom mb-3">
  <div class="container-fluid">
    <a class="navbar-brand" href="dashboard.php"><i class="bi bi-arrow-left-circle me-2"></i>Anasayfa</a>
    <div class="d-flex align-items-center gap-2">
      <span class="small text-secondary">Kullanıcı: <strong><?= htmlspecialchars($user['name']) ?></strong></span>
      <a class="btn btn-outline-secondary btn-sm" href="logout.php"><i class="bi bi-box-arrow-right me-1"></i>Çıkış</a>
    </div>
  </div>
</nav>

<div class="container">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <h4 class="mb-0"><span class="badge bg-info-subtle text-info rounded-pill me-2">&nbsp;</span>Kardeşler Terzi</h4>
    <div class="d-flex gap-2">
      <button id="btnAddEntry" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square me-1"></i>Satır (Gönderi)</button>
      <button id="btnAddReceipt" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square-dotted me-1"></i>Satır (Tahsilat)</button>
      <button id="btnSave" class="btn btn-primary btn-sm"><i class="bi bi-save2 me-1"></i>Kaydet</button>
    </div>
  </div>

  <div class="row g-3 align-items-start">
    <!-- Sol kolon: Gönderiler + Tahsilatlar -->
    <div class="col-12 col-lg-9">
      <!-- GÖNDERİLER -->
      <div class="sheet-card p-3 p-md-4 mb-3">
        <div class="zone-title mb-2"><span class="section-head">Gönderiler</span></div>
        <div class="table-responsive">
          <table class="table grid align-middle mb-2" id="tblEntries">
            <thead>
              <tr>
                <th style="width:140px">TARİH</th>
                <th>GÖNDERİLEN</th>
                <th style="width:120px">ADET</th>
                <th style="width:140px">FİYAT</th>
                <th style="width:160px">TOPLAM TUTAR</th>
                <th style="width:48px"></th>
              </tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr>
                <td colspan="4" class="text-end fw-600">TOPLAM</td>
                <td class="totals" id="entriesTotal">0</td>
                <td></td>
              </tr>
            </tfoot>
          </table>
        </div>
      </div>

      <!-- TAHSİLATLAR -->
      <div class="sheet-card p-3 p-md-4">
        <div class="zone-title mb-2"><span class="section-head">Tahsilatlar</span></div>
        <div class="table-responsive">
          <table class="table grid align-middle mb-2" id="tblReceipts">
            <thead>
              <tr>
                <th style="width:140px">TARİH</th>
                <th>AÇIKLAMA</th>
                <th style="width:160px">TUTAR</th>
                <th style="width:48px"></th>
              </tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr>
                <td colspan="2" class="text-end fw-600">TOPLAM</td>
                <td class="totals" id="receiptsTotal">0</td>
                <td></td>
              </tr>
            </tfoot>
          </table>
        </div>
      </div>
    </div>

    <!-- Sağ kolon: Bakiye -->
    <div class="col-12 col-lg-3">
      <div class="sheet-card p-3 p-md-4">
        <div class="zone-title mb-2"><span class="section-head">Bakiye</span></div>
        <div class="display-6" id="balance">0</div>
        <div class="small text-secondary">Bakiye = Gönderiler Toplamı − Tahsilatlar Toplamı</div>
      </div>
    </div>
  </div>
</div>

<script>
const csrfToken   = "<?= htmlspecialchars($csrf) ?>";
const initialEntries  = <?= json_encode($entries, JSON_UNESCAPED_UNICODE) ?>;
const initialReceipts = <?= json_encode($receipts, JSON_UNESCAPED_UNICODE) ?>;

// ---------- TR format/parsing ----------
const nf  = new Intl.NumberFormat('tr-TR', {minimumFractionDigits:2, maximumFractionDigits:2});
const fmt = n => nf.format(Number.isFinite(+n) ? +n : 0);
function toNum(v){
  if (v == null) return 0;
  v = String(v).trim().replace(/\s/g,'');
  if (v === '') return 0;
  if (v.includes(',') && v.includes('.')) v = v.replace(/\./g,'').replace(',', '.');
  else if (v.includes(',')) v = v.replace(',', '.');
  v = v.replace(/[^0-9.+-]/g,'');
  const n = parseFloat(v);
  return Number.isFinite(n) ? n : 0;
}

// ---------- Satır ekle/sil ----------
function addEntryRow(row={date:'', sent:'', qty:'', price:'', amount:''}){
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td><input type="date" value="${row.date||''}"></td>
    <td><input type="text" placeholder="Gönderilen (ör: TAKIM)" value="${row.sent||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.qty||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.price||''}"></td>
    <td><input type="text" value="${row.amount||''}" readonly></td>
    <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del"><i class="bi bi-x-lg"></i></button></td>
  `;
  document.querySelector('#tblEntries tbody').appendChild(tr);
}
function addReceiptRow(row={date:'', desc:'', amount:''}){
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td><input type="date" value="${row.date||''}"></td>
    <td><input type="text" placeholder="Açıklama" value="${row.desc||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.amount||''}"></td>
    <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del"><i class="bi bi-x-lg"></i></button></td>
  `;
  document.querySelector('#tblReceipts tbody').appendChild(tr);
}

// ---------- Hesaplamalar ----------
function recalc(){
  // Üst toplam
  let eTot = 0;
  document.querySelectorAll('#tblEntries tbody tr').forEach(tr=>{
    const qty   = toNum(tr.children[2].querySelector('input').value);
    const price = toNum(tr.children[3].querySelector('input').value);
    const amt   = Math.round((qty*price + Number.EPSILON) * 100) / 100;
    tr.children[4].querySelector('input').value = fmt(amt);
    eTot += amt;
  });
  document.getElementById('entriesTotal').textContent = fmt(eTot);

  // Alt toplam
  let rTot = 0;
  document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const amt = toNum(tr.children[2].querySelector('input').value);
    rTot += amt;
  });
  document.getElementById('receiptsTotal').textContent = fmt(rTot);

  // Bakiye
  document.getElementById('balance').textContent = fmt(eTot - rTot);
}

function collect(){
  const entries = [];
  document.querySelectorAll('#tblEntries tbody tr').forEach(tr=>{
    const d = tr.children[0].querySelector('input').value;
    const sent = tr.children[1].querySelector('input').value.trim();
    const qty  = toNum(tr.children[2].querySelector('input').value);
    const price= toNum(tr.children[3].querySelector('input').value);
    if(d && sent && qty>0){ entries.push({date:d, sent, qty, price}); }
  });
  const receipts = [];
  document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const d = tr.children[0].querySelector('input').value;
    const desc = tr.children[1].querySelector('input').value.trim();
    const amount = toNum(tr.children[2].querySelector('input').value);
    if(d && desc && amount>0){ receipts.push({date:d, desc, amount}); }
  });
  return {entries, receipts};
}

async function saveAll(){
  const payload = collect();
  const res = await fetch('page2.php?action=save', {
    method:'POST',
    headers:{'Content-Type':'application/json','X-CSRF-Token':csrfToken},
    body: JSON.stringify(payload)
  });
  if(!res.ok){ alert('Kaydetme hatası'); return; }
  const j = await res.json();
  if(j.ok){ alert('Kaydedildi'); } else { alert('Hata: '+(j.msg||'')); }
}

// ---------- Eventler ----------
document.getElementById('btnAddEntry').addEventListener('click', ()=> addEntryRow());
document.getElementById('btnAddReceipt').addEventListener('click', ()=> addReceiptRow());
document.getElementById('btnSave').addEventListener('click', saveAll);

document.addEventListener('input', e=>{
  if(e.target.closest('#tblEntries') || e.target.closest('#tblReceipts')) recalc();
});
document.addEventListener('click', e=>{
  const del = e.target.closest('.row-del');
  if(del){
    e.preventDefault();
    if(confirm('Satırı silmek istiyor musunuz?')){
      del.closest('tr').remove(); recalc();
    }
  }
});

// ---------- Init ----------
(initialEntries.length ? initialEntries : [{date:'',sent:'',qty:'',price:'',amount:''}]).forEach(addEntryRow);
(initialReceipts.length ? initialReceipts : [{date:'',desc:'',amount:''}]).forEach(addReceiptRow);
recalc();
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
