<?php
// page3.php — Miray Çarşı (Miray Müşteriler: satış + tahsilat + bakiye, müşteri ekle/sil + BORÇLAR paneli)
declare(strict_types=1);
session_start();
require_once __DIR__ . '/db.php';

if (empty($_SESSION['user'])) { header('Location: login.php'); exit; }

if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); }
$csrf = $_SESSION['csrf_token'];
$user = $_SESSION['user'];

$flash = $_SESSION['flash'] ?? null; unset($_SESSION['flash']);
function flash($type,$msg){ $_SESSION['flash']=['type'=>$type,'msg'=>$msg]; }

// ------------ Miray müşteri ekle / sil ------------
// SADECE form gönderimleri (action varsa) için çalıştır
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['action'])) {
  $action = $_POST['action'];
  if (!isset($_POST['csrf_token']) || !hash_equals($csrf, (string)$_POST['csrf_token'])) {
    flash('danger','Güvenlik doğrulaması başarısız.'); header('Location: page3.php'); exit;
  }

  if ($action === 'miray_add_customer') {
    $name  = trim((string)($_POST['name'] ?? ''));
    $email = trim((string)($_POST['email'] ?? ''));
    $phone = trim((string)($_POST['phone'] ?? ''));
    $note  = trim((string)($_POST['note'] ?? ''));
    if ($name==='' || mb_strlen($name)<2) { flash('warning','Ad en az 2 karakter olmalı.'); header('Location: page3.php'); exit; }
    if ($email!=='' && !filter_var($email,FILTER_VALIDATE_EMAIL)) { flash('warning','E-posta geçersiz.'); header('Location: page3.php'); exit; }
    $st = $pdo->prepare("INSERT INTO miray_customers (name,email,phone,note) VALUES (:n,:e,:p,:no)");
    $st->execute([':n'=>$name, ':e'=>$email!==''?$email:null, ':p'=>$phone!==''?$phone:null, ':no'=>$note!==''?$note:null]);
    flash('success','Miray müşterisi eklendi.'); header('Location: page3.php'); exit;
  }

  if ($action === 'miray_delete_customer') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id<=0){ flash('warning','Geçersiz müşteri.'); header('Location: page3.php'); exit; }
    try{
      $pdo->beginTransaction();
      $pdo->prepare("DELETE FROM miray_customers WHERE id=:id LIMIT 1")->execute([':id'=>$id]); // ON DELETE CASCADE
      $pdo->commit(); flash('success','Müşteri silindi.');
    }catch(Throwable $e){ $pdo->rollBack(); flash('danger','Silme sırasında hata.'); }
    header('Location: page3.php'); exit;
  }
}

// ------------ Müşteriler + aktif müşteri ------------
$customers = $pdo->query("SELECT id,name FROM miray_customers ORDER BY name ASC")->fetchAll();
$cid = isset($_GET['cid']) ? (int)$_GET['cid'] : (isset($customers[0]['id']) ? (int)$customers[0]['id'] : 0);
$activeCustomer = null; foreach ($customers as $c){ if ((int)$c['id']===$cid) { $activeCustomer=$c; break; } }

// ------------ SAVE endpoint (JSON) ------------
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_GET['action'] ?? '')==='save') {
  header('Content-Type: application/json; charset=utf-8');

  if (empty($_SESSION['user'])) { http_response_code(401); echo json_encode(['ok'=>false,'msg'=>'AUTH']); exit; }
  if (!$activeCustomer) { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Önce müşteri ekleyin.']); exit; }
  if (!isset($_SERVER['HTTP_X_CSRF_TOKEN']) || !hash_equals($csrf, (string)$_SERVER['HTTP_X_CSRF_TOKEN'])) {
    http_response_code(403); echo json_encode(['ok'=>false,'msg'=>'CSRF']); exit;
  }

  $payload = json_decode(file_get_contents('php://input'), true);
  if (!is_array($payload)) { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Bad JSON']); exit; }

  $salesIn = $payload['sales'] ?? [];
  $recsIn  = $payload['receipts'] ?? [];

  $sales = [];
  foreach ($salesIn as $r){
    $d=$r['date']??''; $item=trim((string)($r['item']??'')); $qty=(float)($r['qty']??0); $price=(float)($r['price']??0);
    if($d && $item!=='' && $qty>0){ $sales[] = [$cid,$d,$item,$qty,$price, round($qty*$price,2)]; }
  }
  $receipts = [];
  foreach ($recsIn as $r){
    $d=$r['date']??''; $desc=trim((string)($r['desc']??'')); $amt=(float)($r['amount']??0);
    if($d && $desc!=='' && $amt>0){ $receipts[] = [$cid,$d,$desc,$amt]; }
  }

  $pdo->beginTransaction();
  try{
    $pdo->prepare("DELETE FROM miray_sales WHERE customer_id=?")->execute([$cid]);
    $pdo->prepare("DELETE FROM miray_receipts WHERE customer_id=?")->execute([$cid]);
    if ($sales){
      $ins = $pdo->prepare("INSERT INTO miray_sales (customer_id,s_date,item,qty,price,amount) VALUES (?,?,?,?,?,?)");
      foreach($sales as $row){ $ins->execute($row); }
    }
    if ($receipts){
      $ins = $pdo->prepare("INSERT INTO miray_receipts (customer_id,r_date,description,amount) VALUES (?,?,?,?)");
      foreach($receipts as $row){ $ins->execute($row); }
    }
    $pdo->commit(); echo json_encode(['ok'=>true]);
  }catch(Throwable $e){ $pdo->rollBack(); http_response_code(500); echo json_encode(['ok'=>false,'msg'=>'DB error']); }
  exit;
}

// ------------ BORÇLAR JSON endpoint (AJAX için) ------------
if (($_GET['action'] ?? '') === 'debts') {
  header('Content-Type: application/json; charset=utf-8');
  $stmt = $pdo->query("
    SELECT c.id,c.name,
           COALESCE(s.total_sales,0) AS sales_total,
           COALESCE(r.total_receipts,0) AS receipts_total,
           COALESCE(s.total_sales,0) - COALESCE(r.total_receipts,0) AS balance
    FROM miray_customers c
    LEFT JOIN (SELECT customer_id,SUM(amount) total_sales FROM miray_sales GROUP BY customer_id) s ON s.customer_id=c.id
    LEFT JOIN (SELECT customer_id,SUM(amount) total_receipts FROM miray_receipts GROUP BY customer_id) r ON r.customer_id=c.id
    ORDER BY c.name ASC
  ");
  $rows = $stmt->fetchAll();
  $total = 0.0; foreach ($rows as $d){ $total += (float)$d['balance']; }
  echo json_encode(['ok'=>true, 'rows'=>$rows, 'total'=>$total], JSON_UNESCAPED_UNICODE);
  exit;
}

// ------------ Aktif müşterinin verileri ------------
$salesData = $receiptsData = [];
if ($activeCustomer) {
  $st = $pdo->prepare("SELECT s_date AS date, item, qty, price, amount FROM miray_sales WHERE customer_id=? ORDER BY s_date ASC, id ASC");
  $st->execute([$cid]); $salesData = $st->fetchAll();
  $st = $pdo->prepare("SELECT r_date AS date, description AS `desc`, amount FROM miray_receipts WHERE customer_id=? ORDER BY r_date ASC, id ASC");
  $st->execute([$cid]); $receiptsData = $st->fetchAll();
}

// ------------ BORÇLAR paneli verileri (ilk render için) ------------
$debtsStmt = $pdo->query("
  SELECT c.id,c.name,
         COALESCE(s.total_sales,0) AS sales_total,
         COALESCE(r.total_receipts,0) AS receipts_total,
         COALESCE(s.total_sales,0) - COALESCE(r.total_receipts,0) AS balance
  FROM miray_customers c
  LEFT JOIN (SELECT customer_id,SUM(amount) total_sales FROM miray_sales GROUP BY customer_id) s ON s.customer_id=c.id
  LEFT JOIN (SELECT customer_id,SUM(amount) total_receipts FROM miray_receipts GROUP BY customer_id) r ON r.customer_id=c.id
  ORDER BY c.name ASC
");
$debtsData = $debtsStmt->fetchAll();
$debtsTotal = 0.0; foreach ($debtsData as $d){ $debtsTotal += (float)$d['balance']; }
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Miray Çarşı</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
<style>
  body{background:#f7fafc;}
  .sheet-card{border-radius:16px;border:1px solid rgba(0,0,0,.06);box-shadow:0 10px 30px rgba(2,8,23,.06);}
  .zone-title{font-weight:700;letter-spacing:.3px}
  .grid th, .grid td{vertical-align:middle}
  .grid thead th{background:#e5e7eb; position:sticky; top:0; z-index:2}
  .grid input{border:0;background:transparent; width:100%; padding:.35rem .5rem; outline:none}
  .grid input:focus{background:#fffbe6; box-shadow: inset 0 0 0 1px #f59e0b; border-radius:8px}
  .section-head{background:#e5e7eb;border-radius:10px;padding:.3rem .6rem;display:inline-block}
  .totals{font-weight:700}
  .sticky-tabs{position:sticky; bottom:0; background:#ffffff;border-top:1px solid rgba(0,0,0,.06); padding:.4rem}
  #debtsPanel .pos-neg{font-weight:600}
  #debtsPanel .positive{color:#0ea5e9}
  #debtsPanel .negative{color:#ef4444}
</style>
</head>
<body>
<nav class="navbar navbar-light bg-white border-bottom mb-3">
  <div class="container-fluid">
    <a class="navbar-brand" href="dashboard.php"><i class="bi bi-arrow-left-circle me-2"></i>Anasayfa</a>
    <div class="d-flex align-items-center gap-3">
      <span class="small text-secondary">Kullanıcı: <strong><?= htmlspecialchars($user['name']) ?></strong></span>
    </div>
  </div>
</nav>

<div class="container">
  <!-- Üst başlık ve aksiyonlar (detailWrap içinde) -->
  <div id="detailWrap">
    <div class="d-flex align-items-center justify-content-between mb-3">
      <h4 class="mb-0"><span class="badge bg-primary-subtle text-primary rounded-pill me-2">&nbsp;</span>Miray Çarşı</h4>
      <div class="d-flex gap-2">
        <button id="btnAddSale" class="btn btn-outline-secondary btn-sm" <?= $activeCustomer?'':'disabled' ?>><i class="bi bi-plus-square me-1"></i>Satır (Satış)</button>
        <button id="btnAddRec" class="btn btn-outline-secondary btn-sm" <?= $activeCustomer?'':'disabled' ?>><i class="bi bi-plus-square-dotted me-1"></i>Satır (Tahsilat)</button>
        <button id="btnSave" class="btn btn-primary btn-sm" <?= $activeCustomer?'':'disabled' ?>><i class="bi bi-save2 me-1"></i>Kaydet</button>
      </div>
    </div>

    <?php if ($flash): ?>
      <div class="alert alert-<?= htmlspecialchars($flash['type']) ?>"><?= htmlspecialchars($flash['msg']) ?></div>
    <?php endif; ?>

    <?php if (!$activeCustomer): ?>
      <div class="alert alert-info">Henüz “Miray Müşteriler” eklenmemiş. Aşağıdaki formdan müşteri ekleyin.</div>
    <?php endif; ?>

    <div class="row g-3 align-items-start">
      <!-- Sol: Satış + Tahsilat -->
      <div class="col-12 col-lg-9">
        <div class="sheet-card p-3 p-md-4 mb-3">
          <div class="zone-title mb-2"><span class="section-head">Satışlar</span></div>
          <div class="table-responsive">
            <table class="table grid align-middle mb-2" id="tblSales">
              <thead>
                <tr>
                  <th style="width:140px">Tarih</th>
                  <th>Malzeme Adı</th>
                  <th style="width:120px">Adet</th>
                  <th style="width:140px">Tutar</th>
                  <th style="width:160px">Toplam Tutar</th>
                  <th style="width:48px"></th>
                </tr>
              </thead>
              <tbody></tbody>
              <tfoot>
                <tr>
                  <td colspan="4" class="text-end fw-600">TOPLAM</td>
                  <td class="totals" id="salesTotal">0</td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>

        <div class="sheet-card p-3 p-md-4">
          <div class="zone-title mb-2"><span class="section-head">Tahsilatlar</span></div>
          <div class="table-responsive">
            <table class="table grid align-middle mb-2" id="tblReceipts">
              <thead>
                <tr>
                  <th style="width:140px">Tarih</th>
                  <th>Açıklama</th>
                  <th style="width:160px">Tutar</th>
                  <th style="width:48px"></th>
                </tr>
              </thead>
              <tbody></tbody>
              <tfoot>
                <tr>
                  <td colspan="2" class="text-end fw-600">TOPLAM</td>
                  <td class="totals" id="recTotal">0</td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>
      </div>

      <!-- Sağ: Bakiye -->
      <div class="col-12 col-lg-3">
        <div class="sheet-card p-3 p-md-4">
          <div class="zone-title mb-2"><span class="section-head">Bakiye</span></div>
          <div class="display-6" id="balance">0</div>
          <div class="small text-secondary">Bakiye = Satışlar Toplamı − Tahsilatlar Toplamı</div>
        </div>
      </div>
    </div>
  </div> <!-- /detailWrap -->

  <!-- BORÇLAR Paneli (başlangıçta gizli) -->
  <div id="debtsPanel" class="sheet-card p-3 p-md-4 mt-3 d-none">
    <div class="d-flex align-items-center justify-content-between mb-2">
      <div class="zone-title"><span class="section-head">BORÇLAR</span></div>
      <small class="text-secondary">Tüm Miray müşterilerinin bakiye listesi</small>
    </div>
    <div class="table-responsive">
      <table class="table align-middle mb-2">
        <thead class="table-light">
          <tr>
            <th style="width:56px">#</th>
            <th>Müşteri</th>
            <th class="text-end">Satışlar</th>
            <th class="text-end">Tahsilatlar</th>
            <th class="text-end">Bakiye</th>
            <th class="text-end" style="width:140px">Git</th>
          </tr>
        </thead>
        <tbody id="debtsBody">
          <?php $i=1; foreach ($debtsData as $d):
            $sales=(float)$d['sales_total']; $recs=(float)$d['receipts_total']; $bal=(float)$d['balance'];
            $cls = $bal>=0 ? 'positive':'negative';
          ?>
          <tr>
            <td><?= $i++ ?></td>
            <td><?= htmlspecialchars($d['name']) ?></td>
            <td class="text-end"><?= number_format($sales,2,',','.') ?></td>
            <td class="text-end"><?= number_format($recs,2,',','.') ?></td>
            <td class="text-end pos-neg <?= $cls ?>"><?= number_format($bal,2,',','.') ?></td>
            <td class="text-end">
              <a class="btn btn-sm btn-outline-primary" href="?cid=<?= (int)$d['id'] ?>">
                <i class="bi bi-box-arrow-in-right me-1"></i>Aç
              </a>
            </td>
          </tr>
          <?php endforeach; ?>
        </tbody>
        <tfoot>
          <tr>
            <th colspan="4" class="text-end">GENEL TOPLAM</th>
            <th class="text-end" id="debtsGrand"><?= number_format($debtsTotal,2,',','.') ?></th>
            <th></th>
          </tr>
        </tfoot>
      </table>
    </div>
  </div>

  <!-- Miray Müşteri Yönetimi -->
  <div class="row g-3 mt-3">
    <div class="col-12 col-lg-6">
      <div class="sheet-card p-3 p-md-4">
        <h5 class="mb-3">Miray Müşteri Ekle</h5>
        <form method="post" class="needs-validation" novalidate>
          <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf) ?>">
          <input type="hidden" name="action" value="miray_add_customer">
          <div class="mb-2">
            <label class="form-label">Ad <span class="text-danger">*</span></label>
            <input name="name" type="text" class="form-control" required minlength="2" placeholder="Örn: Ayşe K.">
            <div class="invalid-feedback">Ad en az 2 karakter.</div>
          </div>
          <div class="mb-2"><label class="form-label">E-posta</label><input name="email" type="email" class="form-control"></div>
          <div class="mb-2"><label class="form-label">Telefon</label><input name="phone" type="text" class="form-control"></div>
          <div class="mb-3"><label class="form-label">Not</label><textarea name="note" rows="2" class="form-control"></textarea></div>
          <div class="d-grid"><button class="btn btn-primary" type="submit"><i class="bi bi-plus-circle me-1"></i>Ekle</button></div>
        </form>
      </div>
    </div>
    <div class="col-12 col-lg-6">
      <div class="sheet-card p-3 p-md-4">
        <h5 class="mb-3">Miray Müşteriler</h5>
        <div class="table-responsive">
          <table class="table align-middle">
            <thead class="table-light"><tr><th>#</th><th>Ad</th><th class="text-end">İşlem</th></tr></thead>
            <tbody>
              <?php if (!$customers): ?>
                <tr><td colspan="3" class="text-center text-secondary">Henüz müşteri yok.</td></tr>
              <?php else: foreach ($customers as $c): ?>
                <tr>
                  <td><?= (int)$c['id'] ?></td>
                  <td><a class="customer-tab" href="?cid=<?= (int)$c['id'] ?>"><?= htmlspecialchars($c['name']) ?></a></td>
                  <td class="text-end">
                    <form method="post" class="d-inline del-form" data-name="<?= htmlspecialchars($c['name']) ?>">
                      <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf) ?>">
                      <input type="hidden" name="action" value="miray_delete_customer">
                      <input type="hidden" name="id" value="<?= (int)$c['id'] ?>">
                      <button type="submit" class="btn btn-sm btn-outline-danger"><i class="bi bi-trash me-1"></i>Sil</button>
                    </form>
                  </td>
                </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>

  <!-- Müşteri Sekmeleri + BORÇLAR sekmesi -->
  <div class="sticky-tabs mt-3">
    <ul class="nav nav-pills flex-wrap gap-1 align-items-center">
      <?php foreach ($customers as $c): ?>
        <li class="nav-item">
          <a class="nav-link customer-tab <?= ((int)$c['id']===$cid)?'active':''; ?>" href="?cid=<?= (int)$c['id'] ?>">
            <?= htmlspecialchars($c['name']) ?>
          </a>
        </li>
      <?php endforeach; ?>
      <li class="nav-item ms-2">
        <a id="tabDebts" class="nav-link btn-light" href="#"><i class="bi bi-journal-text me-1"></i>BORÇLAR</a>
      </li>
    </ul>
  </div>
</div>

<!-- Toast: sağ üst bildirim -->
<div class="position-fixed top-0 end-0 p-3" style="z-index:1080">
  <div id="appToast" class="toast align-items-center text-bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
    <div class="d-flex">
      <div class="toast-body" id="appToastBody">Kaydedildi</div>
      <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Kapat"></button>
    </div>
  </div>
</div>

<script>
const csrfToken = "<?= htmlspecialchars($csrf) ?>";
const cid = <?= (int)$cid ?>;
const hasCustomer = <?= $activeCustomer? 'true':'false' ?>;
const initialSales = <?= json_encode($salesData, JSON_UNESCAPED_UNICODE) ?>;
const initialReceipts = <?= json_encode($receiptsData, JSON_UNESCAPED_UNICODE) ?>;

// ---- TR sayı format/parsing ----
const nf = new Intl.NumberFormat('tr-TR',{minimumFractionDigits:2,maximumFractionDigits:2});
const fmt = n => nf.format(Number.isFinite(+n)?+n:0);
function toNum(v){ if(v==null) return 0; v=String(v).trim().replace(/\s/g,''); if(v==='') return 0;
  if(v.includes(',') && v.includes('.')) v=v.replace(/\./g,'').replace(',', '.'); else if(v.includes(',')) v=v.replace(',', '.');
  v=v.replace(/[^0-9.+-]/g,''); const n=parseFloat(v); return Number.isFinite(n)?n:0; }

// ---- Satır ekleme ----
function addSaleRow(row={date:'',item:'',qty:'',price:'',amount:''}){
  const tr=document.createElement('tr');
  tr.innerHTML=`<td><input type="date" value="${row.date||''}" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" value="${row.item||''}" placeholder="Malzeme" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" inputmode="decimal" value="${row.qty||''}" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" inputmode="decimal" value="${row.price||''}" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" value="${row.amount||''}" readonly></td>
  <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del" ${!hasCustomer?'disabled':''}><i class="bi bi-x-lg"></i></button></td>`;
  document.querySelector('#tblSales tbody').appendChild(tr);
}
function addRecRow(row={date:'',desc:'',amount:''}){
  const tr=document.createElement('tr');
  tr.innerHTML=`<td><input type="date" value="${row.date||''}" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" value="${row.desc||''}" placeholder="Açıklama" ${!hasCustomer?'disabled':''}></td>
  <td><input type="text" inputmode="decimal" value="${row.amount||''}" ${!hasCustomer?'disabled':''}></td>
  <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del" ${!hasCustomer?'disabled':''}><i class="bi bi-x-lg"></i></button></td>`;
  document.querySelector('#tblReceipts tbody').appendChild(tr);
}

// ---- Hesaplamalar ----
function recalc(){
  let sTot=0; document.querySelectorAll('#tblSales tbody tr').forEach(tr=>{
    const qty=toNum(tr.children[2].querySelector('input').value);
    const price=toNum(tr.children[3].querySelector('input').value);
    const amt=Math.round((qty*price+Number.EPSILON)*100)/100;
    tr.children[4].querySelector('input').value=fmt(amt); sTot+=amt; });
  document.getElementById('salesTotal').textContent=fmt(sTot);

  let rTot=0; document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const amt=toNum(tr.children[2].querySelector('input').value); rTot+=amt; });
  document.getElementById('recTotal').textContent=fmt(rTot);

  document.getElementById('balance').textContent=fmt(sTot-rTot);
}
function collectData(){
  const sales=[]; document.querySelectorAll('#tblSales tbody tr').forEach(tr=>{
    const d=tr.children[0].querySelector('input').value;
    const item=tr.children[1].querySelector('input').value.trim();
    const qty=toNum(tr.children[2].querySelector('input').value);
    const price=toNum(tr.children[3].querySelector('input').value);
    if(d && item && qty>0) sales.push({date:d,item,qty,price}); });
  const receipts=[]; document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const d=tr.children[0].querySelector('input').value;
    const desc=tr.children[1].querySelector('input').value.trim();
    const amount=toNum(tr.children[2].querySelector('input').value);
    if(d && desc && amount>0) receipts.push({date:d,desc,amount}); });
  return {sales,receipts};
}

// ---- Toast bildirim yardımcıları ----
function notify(msg, type='success') {
  const toastEl = document.getElementById('appToast');
  const bodyEl  = document.getElementById('appToastBody');
  if (!toastEl || !bodyEl) { alert(msg); return; }
  bodyEl.textContent = msg;
  toastEl.classList.remove('text-bg-success','text-bg-danger','text-bg-warning','text-bg-info');
  const okTypes = ['success','danger','warning','info'];
  toastEl.classList.add('text-bg-' + (okTypes.includes(type)?type:'success'));
  const t = new bootstrap.Toast(toastEl, { delay: 2000 });
  t.show();
}

// ---- BORÇLAR verisini AJAX ile yükle ----
async function loadDebts(){
  try{
    const res = await fetch('page3.php?action=debts', { credentials: 'same-origin' });
    const j = await res.json();
    if(!j.ok) return;
    const tbody = document.getElementById('debtsBody');
    const grand = document.getElementById('debtsGrand');
    tbody.innerHTML = '';
    let i = 1;
    j.rows.forEach(d=>{
      const tr = document.createElement('tr');
      const cls = (parseFloat(d.balance) >= 0) ? 'positive' : 'negative';
      tr.innerHTML = `
        <td>${i++}</td>
        <td>${d.name ? String(d.name) : ''}</td>
        <td class="text-end">${fmt(d.sales_total)}</td>
        <td class="text-end">${fmt(d.receipts_total)}</td>
        <td class="text-end pos-neg ${cls}">${fmt(d.balance)}</td>
        <td class="text-end">
          <a class="btn btn-sm btn-outline-primary" href="?cid=${parseInt(d.id,10)}">
            <i class="bi bi-box-arrow-in-right me-1"></i>Aç
          </a>
        </td>`;
      tbody.appendChild(tr);
    });
    grand.textContent = fmt(j.total);
  }catch(e){
    // sessiz geç, panel yine de eski veriyi gösterebilir
  }
}

async function saveAll(){
  try{
    const res=await fetch(`page3.php?action=save&cid=${cid}`,{
      method:'POST',
      headers:{'Content-Type':'application/json','X-CSRF-Token':csrfToken},
      credentials:'same-origin',
      body:JSON.stringify(collectData())
    });
    const ct = res.headers.get('content-type') || '';
    if (!ct.includes('application/json')) { notify('Oturum süreniz dolmuş olabilir. Lütfen tekrar giriş yapın.','danger'); return; }
    const j=await res.json();
    if(res.ok && j.ok){
      notify('Kaydedildi','success');
      if (!document.getElementById('debtsPanel').classList.contains('d-none')) {
        loadDebts(); // panel açıkken anında güncelle
      }
    } else {
      notify('Hata: '+(j.msg||'Bilinmeyen hata'),'danger');
    }
  }catch(e){
    notify('Ağ hatası','danger');
  }
}

// ---- Eventler ----
document.getElementById('btnAddSale')?.addEventListener('click', ()=>addSaleRow());
document.getElementById('btnAddRec')?.addEventListener('click', ()=>addRecRow());
document.getElementById('btnSave')?.addEventListener('click', saveAll);
document.addEventListener('input', e=>{ if(e.target.closest('#tblSales')||e.target.closest('#tblReceipts')) recalc(); });
document.addEventListener('click', e=>{
  const del=e.target.closest('.row-del'); if(del){ e.preventDefault();
    if(confirm('Satırı silmek istiyor musunuz?')){ del.closest('tr').remove(); recalc(); }
  }
});
document.addEventListener('submit', e=>{ const f=e.target.closest('.del-form'); if(f){
  const name=f.getAttribute('data-name')||'bu müşteri'; if(!confirm(`Müşteri "${name}" silinsin mi?\nBu işlem geri alınamaz.`)){ e.preventDefault(); e.stopPropagation(); } }});

// ---- BORÇLAR sekmesi: sadece borç listesi görünsün + veriyi tazele ----
const debtsPanel=document.getElementById('debtsPanel');
const detailWrap=document.getElementById('detailWrap');
const tabDebts=document.getElementById('tabDebts');
function showDebtsOnly(){
  debtsPanel.classList.remove('d-none'); detailWrap.classList.add('d-none'); tabDebts.classList.add('active');
  document.querySelectorAll('.customer-tab, .nav .nav-link.customer-tab').forEach(a=>a.classList.remove('active'));
  loadDebts(); // <-- tıklandığında güncel veriyi çek
}
function hideDebts(){
  debtsPanel.classList.add('d-none'); detailWrap.classList.remove('d-none'); tabDebts.classList.remove('active');
}
tabDebts?.addEventListener('click', (e)=>{ e.preventDefault(); showDebtsOnly(); });
// müşteri sekmesine geçince detaylar geri gelsin
document.querySelectorAll('.customer-tab').forEach(a=>{ a.addEventListener('click', ()=>{ hideDebts(); }); });

// ---- Init ----
(hasCustomer ? (initialSales.length?initialSales:[{date:'',item:'',qty:'',price:'',amount:''}]) : []).forEach(addSaleRow);
(hasCustomer ? (initialReceipts.length?initialReceipts:[{date:'',desc:'',amount:''}]) : []).forEach(addRecRow);
recalc();

// Bootstrap client validation
(()=>{ 'use strict'; const forms=document.querySelectorAll('.needs-validation');
  Array.from(forms).forEach(f=>f.addEventListener('submit',ev=>{ if(!f.checkValidity()){ev.preventDefault();ev.stopPropagation();} f.classList.add('was-validated');},false));})();
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
